/****************************************************************************
*                                                                           *
*                    Standard Parallel Port (SPP) Debug Tool                *
*                 http://www.senet.com.au/~cpeacock/pdebug.htm              *
*                                                                           *
* Source Code originally written by Craig Peacock <cpeacock@senet.com.au>   *
*                                                                           *
* Modified by Flavio Poletti <poletti@writeme.com> on July 21, 1998:        *
*     indentation ;-)                                                       *
*     extension to DJGPP                                                    *
*     parallel port addresses retrieval                                     *
*     user cannot insert addresses any more                                 *
*     mouse is not turned off on redraws                                    *
*     some code re-adjusting                                                *
*     wait for button release after action                                  *
*                                                                           *
* THIS PROGRAM COMES WITH NO WARRANTY: IF YOU DECIDE TO USE IT, YOU DO IT   *
* AT YOUR OWN RISK.                                                         *
*                                                                           *
* This source code can be freely distributed provided that no modifications *
* are made. Should you have any modifications which you believe would be    *
* useful, send them to Craig Peacock <cpeacock@senet.com.au> for inclusion. *
*                                                                           *
* The source code cannot be sold either in source form or pre-programmed    *
* into a device, without the written permission of the Author.              *
*                                                                           *
****************************************************************************/

#include <dos.h>
#include <conio.h>
#include <process.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

/* If compiling with DJGPP, include also... */
#ifdef DJGPP
#include <sys/farptr.h>
#include <go32.h>
#endif

/* Global variables */

int port;                 /* Currently scanned port  */
int port_positions[4];    /* List of all ports       */
int port_number;          /* Number of present ports */

/* Some useful defines... */

#define setpos(x,y) gotoxy((y)+1,(x)+1)
#define BACKGROUNDCOLOR	BLUE
#define FOREGROUNDCOLOR WHITE
#define DISABLEDCOLOR	LIGHTGRAY

/* Something dependent on the compiler... */

#ifdef DJGPP

#define outport outportw
#define MY_SHORT        short

void getports (void)
{
   unsigned int address;            /* Address of Port */
   int j;
   unsigned long ptraddr = 0x0408;  /* Base Address: segment is zero*/

   port_number = 0;
   for (j = 0; j < 3; j++)
   {
      port_positions[j] = _farpeekw(_dos_ds,ptraddr);
      ptraddr += 2;
      if (port_positions[j])
	 port_number++;
   }
}

#else         /* my else clauses always refer to BORLAND C */

#define MY_SHORT

void getports (void)
{
	unsigned int address;     /* Address of Port */
	int j;
	unsigned MY_SHORT int far *ptraddr;

	ptraddr= (unsigned MY_SHORT int far *)0x00000408;
	port_number = 0;

	for (j = 0; j < 3; j++)
	{
		port_positions[j] = *(ptraddr++);
		if (port_positions[j])
			port_number++;
	}
}

#endif

/* Start of common section */

/* Mouse routines */

/* Mouse initialization */
void mouse(void)
{
   union REGS inregs;
   union REGS outregs;

   /* Mouse reset, initialization */
   inregs.x.ax = 0;
   int86(0x33, &inregs, & outregs);

   /* Show mouse */
   inregs.x.ax = 1;
   int86(0x33, &inregs, & outregs);

   /* Set mouse initial position */
   inregs.x.ax = 4;
   inregs.x.cx = 0;
   inregs.x.dx = 8;
   int86(0x33, &inregs, & outregs);
}

void mouseon(void)
{
   union REGS inregs;
   union REGS outregs;
   inregs.x.ax = 1;
   int86(0x33,&inregs,&outregs);
}

void mouseoff(void)
{
   union REGS inregs;
   union REGS outregs;
   inregs.x.ax = 2;
   int86(0x33,&inregs,&outregs);
}

void updatemouse(void)
{
   union REGS inregs;
   union REGS outregs;

   /* Hide mouse... */
   inregs.x.ax = 2;
   int86(0x33,&inregs,&outregs);

   /* ... and show it again! */
   inregs.x.ax = 1;
   int86(0x33,&inregs,&outregs);
}

void mousestatus
(MY_SHORT int *row,
 MY_SHORT int *column,
 MY_SHORT int *buttonstatus)
{
   union REGS inregs;
   union REGS outregs;
   inregs.x.ax = 3;
   int86(0x33, &inregs, &outregs);
   *column = ((outregs.x.cx)/8)*8;
   *row = ((outregs.x.dx)/8)*8;
   *buttonstatus = outregs.x.bx;
}

/* Other stuff... */

void data(int bit, int port, int base)
{
   if (base & bit)
      base &= ~bit;
   else
      base |= bit;
   outport(port,base);
}

/* Show inactive ports as disabled */

void disable_port(int j)
{
   setpos(6,35+11*j);
   textbackground(BACKGROUNDCOLOR);
   textcolor(DISABLEDCOLOR);
   cprintf("LPT %d [ ]",j+1);
   textcolor(FOREGROUNDCOLOR);
}

/* Actions performed on exit */
void exit_jobs(void)
{
   /* Turn off the mouse */
   mouseoff();

   /* Clear text screen */
   textbackground(0);
   textcolor(7);
   clrscr();

   /* Restore cursor */
   _setcursortype(_NORMALCURSOR);

   /* Send message */
   printf("Parallel Port Debug Tool V2.0 - August 1998\n");
   printf("Source Code Available from http://www.senet.com.au/~cpeacock/pdebug.htm\n\n");
}

/* Initialization */
int init(void)
{
   int j;

   /* See whether there are ports to debug! */
   getports();
   if (!port_number)
   {
      printf("Parallel Port Debug Tool V2.0 - August 1998\n");
      printf("Source Code Available from http://www.senet.com.au/~cpeacock/pdebug.htm\n");
      printf("Sorry, No detectable Parallel Port found on your PC\n\n");
      return(0);
   }

   /* Set default port to LPT1 */
   port = port_positions[0];

   /* Screen initialization */
   textbackground(0);
   textcolor(FOREGROUNDCOLOR);
   clrscr();
   
   textbackground(BACKGROUNDCOLOR);
   cprintf("ͻ");
   cprintf("                                                                              ");
   cprintf("                         PARALLEL PORT DEBUG TOOL V2.0                        ");
   cprintf("                       http://www.senet.com.au/~cpeacock                      ");
   cprintf("                                                                              ");
   cprintf("Ķ");
   cprintf(" Base Address 0x                  LPT 1 [ ]  LPT 2 [ ]  LPT 3 [ ]  LPT 4 [ ]  ");
   cprintf("Ķ");
   cprintf(" Printer Port     Pin 2  Pin 3  Pin 4  Pin 5  Pin 6  Pin 7  Pin 8  Pin 9      ");
   cprintf(" Bit Number         0      1      2      3      4      5      6      7        ");
   cprintf(" Status            [ ]    [ ]    [ ]    [ ]    [ ]    [ ]    [ ]    [ ]       ");
   cprintf("Ķ");
   cprintf(" Base + 1   Status Port  (Input)    Base + 2   Control Port (Open Collector) ");
   cprintf("                                                                             ");
   cprintf(" Bit 7 [ ]  Busy        (Pin 11)    Bit 7 [ ]   Reserved                     ");
   cprintf(" Bit 6 [ ]  ACK (NOT)   (Pin 10)    Bit 6 [ ]   Reserved                     ");
   cprintf(" Bit 5 [ ]  Paper Out   (Pin 12)    Bit 5 [ ]   Enable Bidirectional Port    ");
   cprintf(" Bit 4 [ ]  Select In   (Pin 13)    Bit 4 [ ]   Enable IRQ via ACK (Pin 10)  ");
   cprintf(" Bit 3 [ ]  Error (NOT) (Pin 15)    Bit 3 [ ]   Select Printer     (Pin 17)  ");
   cprintf(" Bit 2 [ ]  IRQ (NOT)               Bit 2 [ ]   Initialize Printer (Pin 16)  ");
   cprintf(" Bit 1 [ ]  Reserved                Bit 1 [ ]   Auto Linefeed      (Pin 14)  ");
   cprintf(" Bit 0 [ ]  Reserved                Bit 0 [ ]   Strobe             (Pin 1)   ");
   cprintf("ͼ");

   /* Show non-existent ports as disabled */
   for (j = port_number; j < 4; j++)
      disable_port(j);

   /* Turn on the mouse */
   mouseon();

   /* Hide the cursor */
   _setcursortype(_NOCURSOR);

   /* Clear output ports */
   outport(port,0);
   outport(port + 2,0);

   /* Ok. */
   return 1;
}

/* Main program */

main()
{
   /* VARIABLES */
   int
      base0,       /* buffer for port + 0 */
      oldbase0,    /* previous value      */
      base1,       /* buffer for port + 1 */
      oldbase1,    /* previous value      */
      base2,       /* buffer for port + 2 */
      oldbase2,    /* previous value      */
      oldport,     /* previous port       */
      c,           /* char buffer         */
      a,           /* temp variable       */
      j,           /* temp & cycle var.   */
      mask;

   MY_SHORT int   /* MY_SHORT assures those variables are 2 bytes */
      row,
      column,
      buttonstatus;

   
   /* INITIALIZATION */
   /* init sends 0 if ports could not be found */
   if (!init())
      return (1);

   /* MAIN LOOP */
   do
   {
      /* Get ports status */
      base0 = inportb(port);
      base1 = inportb(port+1);
      base2 = inportb(port+2);

      /* Freshen port section */
      if (oldport != port)
      {
	 mouseoff();
         setpos(6,17);
         textbackground(BACKGROUNDCOLOR);
         cprintf("%04X (%d)",port,port);
         for (j = 0; j < port_number; j++)
         {
            setpos(6,42+11*j);            
            if (port == port_positions[j])
               cprintf("X");
	    else
	       cprintf(" ");
         }
         oldport=port;
         mouseon();
      }

      /* Freshen base + 0 section */
      if (oldbase0 != base0)
      {
         for (j = 0, mask = 0x01; j < 8; j++, mask <<= 1)
	 {
            setpos(10,21+7*j);
            if (base0 & mask)
               cprintf("X");
            else
               cprintf(" ");
         }
         oldbase0 = base0;
      }

      /* Freshen base + 1 section */
      if (oldbase1 != base1)
      {
         for (j = 0, mask = 0x80; j < 8; j++, mask >>= 1)
         {
            setpos(14+j,9);
            if (base1 & mask)
               cprintf("X");
	    else
               cprintf(" ");
	 }
         oldbase1 = base1;
      }

      /* Freshen base + 2 section */
      if (oldbase2 != base2)
      {
         for (j = 0, mask = 0x80; j < 8; j++, mask >>= 1)
         {
	    setpos(14+j,45);
	    if (base2 & mask)
               cprintf("X");
            else
               cprintf(" ");
         }         
         oldbase2 = base2;
      }

      /* Function keys input section */
      a = kbhit();
      if (a != 0)
      {
         c = getch();
         setpos(1,0);
         if (!c)        /* Function keys F1...F8 */
         {
            c = getch() - 59;
            if ((c >= 0) && (c <= 7))
	    {
	       /* Find the right mask */
	       mask = 0x01 << c;
               /* Toggle bit */
               if (base0 & mask)
                  base0 &= ~mask;
               else
                  base0 |= mask;
               /* Give it in output */
               outportb(port,base0);
	    }
         }
      }

      /* Mouse input section */
      mousestatus(&row,&column,&buttonstatus);
      if (buttonstatus != 0)
      {
         /* First of all, inspect row controls... */
         if (row == 48)  /* Change port? */
	 {
	    j = column - 336;
	    if ((!(j % 88)) && (j / 88 < 4) && (j >= 0))
               if (port_number > j/88)
                  port = port_positions[j/88];
         }
         else if (row == 80)  /* change data port status? */
         {
            j = column - 168;
            if ((!(j % 56)) && (j / 56 < 8) && (j >= 0))
	       data(0x01<<(j/56),port,base0); /* mask = 0x01 << j */
         }

         /* ... then, inspect column controls... */
         if (column == 360)  /* Change control port status? */
         {
            j = (row - 112)/8;
            if ((j >= 0) && (j < 8))
               data(0x80>>j,port+2,base2);
         }

	 /* Wait for the user to release mouse buttons */
	 while (buttonstatus)
	    mousestatus(&row,&column,&buttonstatus);
      }
   } while (c != 27);

   /* EXIT FUNCTION */
   exit_jobs();

   return 0;
}
